<?php
// ============================================================
// index.php
// - Reads a value from Google Sheets (A2)
// - Appends text to a Google Doc
// - Exports that Doc as PDF using Google Drive API
// - Saves the PDF on the server
// - Lists files in Google Drive root
// - Uploads the created PDF to Google Drive root
// ============================================================

// --------------------------
// 0) PHP error display
// --------------------------
ini_set("display_errors", 1);ini_set("display_startup_errors", 1);error_reporting(E_ALL);
// Small boot log
echo "<hr><b>Arrancando...</b><br>";

// --------------------------
// 1) Paths and basic checks
// --------------------------
$gsheetsDir = __DIR__ . "/Google-Sheets-n-Docs";
$autoloadPath = $gsheetsDir . "/vendor/autoload.php";
$credentialsPath = $gsheetsDir . "/credentials.json";
$configPath = $gsheetsDir . "/config.php";

// Validate directories/files exist
if (!is_dir($gsheetsDir)) {
    echo "<hr><b>Error:</b> No existe la carpeta: " . $gsheetsDir . "<br>";
    exit;
}
if (!file_exists($autoloadPath)) {
    echo "<hr><b>Error:</b> No existe: " . $autoloadPath . "<br>";
    exit;
}
if (!file_exists($credentialsPath)) {
    echo "<hr><b>Error:</b> No existe: " . $credentialsPath . "<br>";
    exit;
}
if (!file_exists($configPath)) {
    echo "<hr><b>Error:</b> No existe: " . $configPath . "<br>";
    exit;
}

echo "autoload OK<br>";
echo "credentials OK<br>";
echo "config OK<br>";

// Load autoload (needed for all Google classes)
require_once $autoloadPath;

// Read service account email (just for debug)
$credRaw = file_get_contents($credentialsPath);
$credJson = json_decode($credRaw, true);

echo "<hr><b>Service Account client_email:</b><br>";
if (isset($credJson["client_email"])) {
    echo $credJson["client_email"] . "<br>";
} else {
    echo "No encontré client_email en credentials.json<br>";
}

// --------------------------
// 2) Main try/catch
// --------------------------
try {
    // ============================================================
    // 2.1) Load Sheets service from your config.php
    //      Your config.php should define $service as Google_Service_Sheets
    // ============================================================
    require_once $configPath;

    if (!isset($service)) {
        echo "<hr><b>Error:</b> config.php no creó \$service.<br>";
        exit;
    }

    // ------------------------------------------------------------
    // 3) Google Sheets - read/write
    // ------------------------------------------------------------
    $spreadsheetId = "1-ey9xm7xidXKEQFsR8Bd8EkDEMsadqDRTyc32mywJ5Y";
    $sheetName = "usuarios";

    echo "<hr><b>Ping: verificar acceso al Spreadsheet</b><br>";

    // Read spreadsheet metadata
    $meta = $service->spreadsheets->get($spreadsheetId);
    echo "Título: " . $meta->getProperties()->getTitle() . "<br>";
    echo "sheetName: " . $sheetName . "<br>";

    echo "<hr><b>1) Escribir \"ok\" en A1</b><br>";

    // Write A1 = ok
    $rangeWrite = $sheetName . "!A1";
    $bodyWrite = new Google_Service_Sheets_ValueRange([
        "values" => [["ok"]]
    ]);
    $paramsWrite = ["valueInputOption" => "RAW"];

    $resultWrite = $service->spreadsheets_values->update(
        $spreadsheetId,
        $rangeWrite,
        $bodyWrite,
        $paramsWrite
    );

    echo "Actualizado: " . $resultWrite->getUpdatedRange() . "<br>";

    echo "<hr><b>2) Leer A2</b><br>";

    // Read A2
    $rangeRead = $sheetName . "!A2";
    $responseRead = $service->spreadsheets_values->get($spreadsheetId, $rangeRead);
    $valuesRead = $responseRead->getValues();

    $valorA2 = "";
    if (!empty($valuesRead) && isset($valuesRead[0][0])) {
        $valorA2 = $valuesRead[0][0];
    }

    echo "Valor en " . $rangeRead . ": <b>" . $valorA2 . "</b><br>";

    echo "<hr><b>3) Borrar (limpiar) B1</b><br>";

    // Clear B1
    $rangeClear = $sheetName . "!B1";
    $clearRequest = new Google_Service_Sheets_ClearValuesRequest();

    $service->spreadsheets_values->clear(
        $spreadsheetId,
        $rangeClear,
        $clearRequest
    );

    echo "Celda limpiada: " . $rangeClear . "<br>";

        // ------------------------------------------------------------
        // 3.2) Example: Insert rows in Google Sheets at row X (1-based)
        // ------------------------------------------------------------
        echo "<hr><b>3.2) Innsertar filas nuevas conn datos</b><br>";

        // Row where you want to insert (X = 1..n)
        $filaX = 5;

        // How many rows to insert
        $numFilas = 2;

        // ------------------------------------------------------------
        // 1) Get the numeric sheetId from the sheet name
        // ------------------------------------------------------------
        $meta = $service->spreadsheets->get($spreadsheetId);

        $sheetId = null;
        foreach ($meta->getSheets() as $sh) {
            $props = $sh->getProperties();
            if ($props->getTitle() == $sheetName) {
                $sheetId = $props->getSheetId();
                break;
            }
        }

        if ($sheetId === null) {
            echo "<hr><b>Error:</b> No encontré el sheetName \"".$sheetName."\".<br>";
            exit;
        }

        // ------------------------------------------------------------
        // 2) Insert rows at row X (API uses 0-based indexes)
        //    Row X (1-based) => startIndex = X - 1
        // ------------------------------------------------------------
        $startIndex = $filaX - 1;
        $endIndex = $startIndex + $numFilas;

        $requests = [
            new Google_Service_Sheets_Request([
                "insertDimension" => [
                    "range" => [
                        "sheetId" => $sheetId,
                        "dimension" => "ROWS",
                        "startIndex" => $startIndex,
                        "endIndex" => $endIndex
                    ],
                    "inheritFromBefore" => false
                ]
            ])
        ];

        $batchUpdateRequest = new Google_Service_Sheets_BatchUpdateSpreadsheetRequest([
            "requests" => $requests
        ]);

        $service->spreadsheets->batchUpdate($spreadsheetId, $batchUpdateRequest);

        echo "<hr><b>Filas insertadas:</b> ".$numFilas." en la fila ".$filaX." (antes de esa fila).<br>";
        
    // ============================================================
    // 4) Google Docs - append text
    // ============================================================
    echo "<hr><b>4) Editar Google Doc (append al final)</b><br>";

    // Document ID (from your URL)
    $documentId = "1pLZFy6HqBw5t5EB06L23x9axhsts4R6yu7WGTncH3mk";

    // Create a new client for Docs + Drive (same service account)
    $clientDocsDrive = new Google_Client();
    $clientDocsDrive->setApplicationName("Sheets + Docs + Drive");
    $clientDocsDrive->setAuthConfig($credentialsPath);

    // Scopes:
    // - Docs: edit doc
    // - Drive: export / list / upload
    $clientDocsDrive->setScopes([
        Google_Service_Docs::DOCUMENTS,
        Google_Service_Drive::DRIVE
    ]);

    // Docs service
    $docsService = new Google_Service_Docs($clientDocsDrive);

    // Text to append
    $textoNuevo = "info leida desde el gsheet celda A2: " . $valorA2 . "\n";

    // Build requests for Docs batchUpdate
    $requests = [
        new Google_Service_Docs_Request([
            "insertText" => [
                "endOfSegmentLocation" => ["segmentId" => ""],
                "text" => $textoNuevo
            ]
        ])
    ];

    $batchUpdateRequest = new Google_Service_Docs_BatchUpdateDocumentRequest([
        "requests" => $requests
    ]);

    // Execute append
    $docsService->documents->batchUpdate($documentId, $batchUpdateRequest);

    echo "Texto agregado correctamente al documento.<br>";

    // ============================================================
    // 5) Export Google Doc as PDF using Google Drive API
    // ============================================================
    echo "<hr><b>5) Exportar Google Doc a PDF (Drive API)</b><br>";

    // Drive service (same client)
    $driveService = new Google_Service_Drive($clientDocsDrive);

    // Ensure exports folder exists
    $exportsDir = __DIR__ . "/exports";
    if (!is_dir($exportsDir)) {
        mkdir($exportsDir, 0777, true);
    }

    // Build output PDF path
    $pdfName = "gdoc_" . $documentId . "_" . date("Ymd_His") . ".pdf";
    $pdfPath = $exportsDir . "/" . $pdfName;

    // Export file: application/pdf
    $exportResponse = $driveService->files->export($documentId, "application/pdf", ["alt" => "media"]);
    
    // Get bytes (compat across versions)
    $pdfBytes = "";
    if (is_string($exportResponse)) {
        $pdfBytes = $exportResponse;
    } else {
        // Usually a PSR-7 response or stream wrapper
        if (method_exists($exportResponse, "getBody")) {
            $pdfBytes = $exportResponse->getBody()->getContents();
        }
    }

    // Save on server
    $saved = file_put_contents($pdfPath, $pdfBytes);

    if ($saved > 0) {
        echo "PDF guardado en servidor: <b>" . $pdfPath . "</b><br>";
        echo "Archivo: <b>" . $pdfName . "</b><br>";
        echo "Link (si aplica): <a href=\"exports/".$pdfName."\" target=\"_blank\">Abrir PDF</a><br>";
    } else {
        echo "<b>Error:</b> No se pudo guardar el PDF en: " . $pdfPath . "<br>";
    }

    // ============================================================
    // 6) List files inside a specific folder (shared with service account)
    // ============================================================
    echo "<hr><b>6) Listar archivos dentro de la carpeta compartida</b><br>";

    // Folder ID provided by you
    $folderIdDestino = "1vGB2YIAliSuWlSpJPhp-DfYYq4hNoP8j";

    $folderInfo = $driveService->files->get($folderIdDestino, [
        "fields" => "id, name, driveId, owners, capabilities",
        "supportsAllDrives" => true
    ]);

    echo "<b>Nombre folder compartido:</b> " . $folderInfo->getName() . "<br>";
    echo "<b>ID:</b> " . $folderInfo->getId() . "<br>";
    echo "<b>driveId:</b> " . $folderInfo->getDriveId() . "<hr>";

    // Small comment: list only direct children of the folder, excluding trashed
    $pageToken = null;
    $totalMostrados = 0;

    do {
        $optParamsList = [
            "q" => "'" . $folderIdDestino . "' in parents and trashed = false",
            "pageSize" => 100,
            "fields" => "nextPageToken, files(id, name, mimeType, size, modifiedTime, webViewLink)",
            "orderBy" => "modifiedTime desc"
        ];

        if ($pageToken != null && $pageToken != "") {
            $optParamsList["pageToken"] = $pageToken;
        }

        $listResponse = $driveService->files->listFiles($optParamsList);
        $files = $listResponse->getFiles();

        if (!empty($files)) {
            foreach ($files as $f) {
                $size = $f->getSize();
                if ($size === null || $size === "") { $size = "N/A"; }

                echo "<b>Nombre:</b> " . $f->getName() . "<br>";
                echo "<b>ID:</b> " . $f->getId() . "<br>";
                echo "<b>MIME:</b> " . $f->getMimeType() . "<br>";
                echo "<b>Tamaño:</b> " . $size . "<br>";
                echo "<b>Modificado:</b> " . $f->getModifiedTime() . "<br>";

                $link = $f->getWebViewLink();
                if ($link != "") {
                    echo "Ver: <a href=\"" . $link . "\" target=\"_blank\">Abrir</a><br>";
                }

                echo "<hr>";
                $totalMostrados++;
            }
        }

        $pageToken = $listResponse->getNextPageToken();

    } while ($pageToken != null && $pageToken != "");

    if ($totalMostrados == 0) {
        echo "No encontré archivos en esa carpeta o no tengo permisos.<br>";
        echo "Verifica que la carpeta esté compartida con: <b>" . $credJson["client_email"] . "</b><br>";
    }

    // ============================================================
    // 7) Upload the just-created PDF into that folder
    // ============================================================
    echo "<hr><b>7) Subir el PDF creado a la carpeta (requiere OAuth usuario)</b><br>";

    function getDriveServiceUsuarioOAuth($oauthCredPath, $tokenPath) {
        $client = new Google_Client();
        $client->setAuthConfig($oauthCredPath);

        // Small comment: redirect back to this same script URL
        $redirectUri = (isset($_SERVER["HTTPS"]) ? "https" : "http") . "://" . $_SERVER["HTTP_HOST"] . $_SERVER["PHP_SELF"];
        $client->setRedirectUri($redirectUri);

        // Small comment: offline keeps a refresh token so you don't login every time
        $client->setAccessType("offline");
        $client->setPrompt("consent");

        // Small comment: play with the profiles depending what you need
        $client->setScopes([Google_Service_Drive::DRIVE]);//Conntrol total de GDrive
            //Google_Service_Drive::DRIVE_FILE //este permite editar/borrar/crear solo los que crea este script pero no listar
            //Google_Service_Drive::DRIVE_READONLY //este permite solo listar pero nno editar,etc
        
        // Small comment: handle OAuth callback
        if (isset($_GET["code"]) && $_GET["code"] != "") {
            $token = $client->fetchAccessTokenWithAuthCode($_GET["code"]);
            file_put_contents($tokenPath, json_encode($token));
            echo "<hr><b>OAuth OK:</b> token guardado. Recarga la página sin ?code.<br>";
            exit;
        }

        // Load existing token
        if (file_exists($tokenPath)) {
            $token = json_decode(file_get_contents($tokenPath), true);
            $client->setAccessToken($token);
        }

        // Refresh token if expired, else ask user to authorize
        if ($client->isAccessTokenExpired()) {
            $refresh = $client->getRefreshToken();
            if ($refresh != "") {
                $client->fetchAccessTokenWithRefreshToken($refresh);
                file_put_contents($tokenPath, json_encode($client->getAccessToken()));
            } else {
                $authUrl = $client->createAuthUrl();
                echo "<hr><b>Autoriza Drive (1 vez):</b><br>";
                echo "<a href=\"" . $authUrl . "\">Click para autorizar</a><br>";
                exit;
            }
        }

        return new Google_Service_Drive($client);
    }
    
    // Folder ID (tu carpeta compartida en Mi unidad)
    $folderIdDestino = "1vGB2YIAliSuWlSpJPhp-DfYYq4hNoP8j";

    // Rutas en tu servidor
    $oauthCredPath = __DIR__ . "/Google-Sheets-n-Docs/oauth_credentials.json";
    $tokenPath = __DIR__ . "/Google-Sheets-n-Docs/token_drive.json";

    if (!file_exists($oauthCredPath)) {
        echo "<b>Error:</b> No existe oauth_credentials.json en: " . $oauthCredPath . "<br>";
    } else if (!file_exists($pdfPath)) {
        echo "<b>Error:</b> No existe el PDF en servidor para subir: " . $pdfPath . "<br>";
    } else {
        $driveUsuario = getDriveServiceUsuarioOAuth($oauthCredPath, $tokenPath);

        // Small comment: parent points to your folder, not root
        $fileMetadata = new Google_Service_Drive_DriveFile([
            "name" => $pdfName,
            "parents" => [$folderIdDestino]
        ]);

        $content = file_get_contents($pdfPath);

        $uploadedFile = $driveUsuario->files->create(
            $fileMetadata,
            [
                "data" => $content,
                "mimeType" => "application/pdf",
                "uploadType" => "multipart",
                "fields" => "id, name, webViewLink"
            ]
        );

        echo "PDF subido correctamente a la carpeta.<br>";
        echo "<b>Nombre:</b> " . $uploadedFile->getName() . "<br>";
        echo "<b>ID:</b> " . $uploadedFile->getId() . "<br>";

        $webViewLink = $uploadedFile->getWebViewLink();
        if ($webViewLink != "") {
            echo "Ver en Drive: <a href=\"" . $webViewLink . "\" target=\"_blank\">Abrir</a><br>";
        }
    }

    echo "<hr><b>A) Listar archivos en raíz (Mi unidad) usando OAuth</b><br>";

    $oauthCredPath = __DIR__ . "/Google-Sheets-n-Docs/oauth_credentials.json";
    $tokenPath = __DIR__ . "/Google-Sheets-n-Docs/token_drive.json";

    $driveUsuario = getDriveServiceUsuarioOAuth($oauthCredPath, $tokenPath);

    // Small comment: list direct children of root (My Drive)
    $optParamsRoot = [
        "q" => "'root' in parents and trashed = false",
        "pageSize" => 100,
        "fields" => "files(id, name, mimeType, size, modifiedTime, webViewLink)",
        "orderBy" => "modifiedTime desc"
    ];

    $listRoot = $driveUsuario->files->listFiles($optParamsRoot);
    $filesRoot = $listRoot->getFiles();

    if (empty($filesRoot)) {
        echo "No encontré archivos en raíz.<br>";
    } else {
        foreach ($filesRoot as $f) {
            $size = $f->getSize();
            if ($size === null || $size === "") { $size = "N/A"; }

            echo "<b>Nombre:</b> " . $f->getName() . "<br>";
            echo "<b>ID:</b> " . $f->getId() . "<br>";
            echo "<b>MIME:</b> " . $f->getMimeType() . "<br>";
            echo "<b>Tamaño:</b> " . $size . "<br>";
            echo "<b>Modificado:</b> " . $f->getModifiedTime() . "<br>";

            $link = $f->getWebViewLink();
            if ($link != "") {
                echo "Ver: <a href=\"" . $link . "\" target=\"_blank\">Abrir</a><br>";
            }
            echo "<hr>";
        }
    }

    echo "<hr><b>B) Mandar a papelera el archivo por ID (OAuth)</b><br>";

    $fileIdBorrar = "1Is9LsUIcyKejGpiMxcpPePfAm8tHLixM";

    $driveUsuario = getDriveServiceUsuarioOAuth($oauthCredPath, $tokenPath);

    $driveUsuario->files->update(
        $fileIdBorrar,
        new Google_Service_Drive_DriveFile(["trashed" => true]),
        ["fields" => "id, trashed"]
    );

    echo "Archivo mandado a papelera: <b>" . $fileIdBorrar . "</b><br>";

    echo "<hr><b>C) Crear Google Sheet \"prueba\" y moverlo a la carpeta</b><br>";

    $folderIdDestino = "1vGB2YIAliSuWlSpJPhp-DfYYq4hNoP8j";

    $oauthCredPath = __DIR__ . "/Google-Sheets-n-Docs/oauth_credentials.json";
    $tokenPath = __DIR__ . "/Google-Sheets-n-Docs/token_drive.json";

    // Small comment: build OAuth client for Sheets using the same saved token
    $clientSheetsOAuth = new Google_Client();
    $clientSheetsOAuth->setAuthConfig($oauthCredPath);

    $redirectUri = (isset($_SERVER["HTTPS"]) ? "https" : "http") . "://" . $_SERVER["HTTP_HOST"] . $_SERVER["PHP_SELF"];
    $clientSheetsOAuth->setRedirectUri($redirectUri);

    $clientSheetsOAuth->setAccessType("offline");
    $clientSheetsOAuth->setPrompt("consent");

    // Small comment: need Sheets scope to create spreadsheets
    $clientSheetsOAuth->setScopes([Google_Service_Sheets::SPREADSHEETS]);

    if (file_exists($tokenPath)) {
        $token = json_decode(file_get_contents($tokenPath), true);
        $clientSheetsOAuth->setAccessToken($token);
    }

    if ($clientSheetsOAuth->isAccessTokenExpired()) {
        $refresh = $clientSheetsOAuth->getRefreshToken();
        if ($refresh != "") {
            $clientSheetsOAuth->fetchAccessTokenWithRefreshToken($refresh);
            file_put_contents($tokenPath, json_encode($clientSheetsOAuth->getAccessToken()));
        } else {
            $authUrl = $clientSheetsOAuth->createAuthUrl();
            echo "<hr><b>Autoriza Sheets (1 vez):</b><br>";
            echo "<a href=\"" . $authUrl . "\">Click para autorizar</a><br>";
            exit;
        }
    }

    $sheetsOAuth = new Google_Service_Sheets($clientSheetsOAuth);

    $spreadsheet = new Google_Service_Sheets_Spreadsheet([
        "properties" => ["title" => "prueba"]
    ]);

    // Small comment: create spreadsheet (it will start in My Drive root)
    $created = $sheetsOAuth->spreadsheets->create($spreadsheet);
    $newSheetId = $created->getSpreadsheetId();

    echo "Sheet creado.<br>";
    echo "<b>ID:</b> " . $newSheetId . "<br>";

    // Small comment: now move it to the target folder using Drive OAuth
    $driveUsuario = getDriveServiceUsuarioOAuth($oauthCredPath, $tokenPath);

    $fileInfo = $driveUsuario->files->get($newSheetId, ["fields" => "parents"]);
    $prevParentsArr = $fileInfo->getParents();
    $prevParents = "";
    if (!empty($prevParentsArr)) {
        $prevParents = implode(",", $prevParentsArr);
    }

    $driveUsuario->files->update(
        $newSheetId,
        new Google_Service_Drive_DriveFile(),
        [
            "addParents" => $folderIdDestino,
            "removeParents" => $prevParents,
            "fields" => "id, parents, webViewLink"
        ]
    );

    echo "Movido a la carpeta destino: <b>" . $folderIdDestino . "</b><br>";
    echo "Abrir: <a href=\"" . $created->getSpreadsheetUrl() . "\" target=\"_blank\">Ver Sheet</a><br>";

echo "<hr><b>Listo.</b><br>";

} catch (Google_Service_Exception $e) {
    echo "<hr><b>Google_Service_Exception</b><br>";
    echo "HTTP Code: " . $e->getCode() . "<br>";
    echo "<b>Message:</b><br><pre>" . $e->getMessage() . "</pre><br>";

    $errors = $e->getErrors();
    if (!empty($errors)) {
        echo "<b>Errors:</b><br><pre>";
        var_dump($errors);
        echo "</pre><br>";
    }

    echo "<hr>";
    exit;

} catch (Exception $e) {
    echo "<hr><b>Exception</b><br>";
    echo "Code: " . $e->getCode() . "<br>";
    echo "<b>Message:</b><br><pre>" . $e->getMessage() . "</pre><br>";
    echo "<hr>";
    exit;
}
