(function (){
	"use strict";
	
	const EMPTY = "===empty===";
	const DEFAULT_FPS = 60;
	const DT_FPS = 1000 / DEFAULT_FPS;
	const MIN = 0;
	const MAX = 100;
	const UNIT_PATTERN =
		/(px|%|vh|vw|em|rem|pt|cm|mm|in|pc|ex|ch|vmin|vmax|lh|rlh|vb|vi|svw|svh|lvw|lvh|dvw|dvh)/g;
	
	const lerp = (p1, p2, t) => {
		return p1 + (p2 - p1) * t;
	};
	
	function clamp(value, min, max) {
		return Math.min(Math.max(value, min), max);
	}
	
	function each(array, callback) {
		for (let i = 0; i < array.length; i++) {
			callback(array[i], i, array);
		}
	}
	
	function filter(array, callback) {
		const result = [];
		for (let i = 0; i < array.length; i++) {
			if (callback(array[i], i, array)) {
				result.push(array[i]);
			}
		}
		return result;
	}
	
	function reduce(array, callback, initialValue) {
		let accumulator = initialValue;
		for (let i = 0; i < array.length; i++) {
			accumulator = callback(accumulator, array[i], i, array);
		}
		return accumulator;
	}
	
	class FrameManager {
		frameId;
		keepAliveFrameId;
		lastTimestamp;
		frames;
		keepAliveFrames;
		defaultTimestep;
		
		constructor() {
			this.frameId = null;
			this.keepAliveFrameId = null;
			this.lastTimestamp = null;
			this.frames = [];
			this.keepAliveFrames = [];
			this.defaultTimestep = (1 / 60) * 1000;
		}
		
		handleFrameLoop = (timestamp) => {
			if (this.lastTimestamp) {
				const delta = timestamp - this.lastTimestamp;
				this.keepAliveFrames.forEach((frame) =>
					frame.call(this, { delta, timestamp })
				);
			}
			this.lastTimestamp = timestamp;
			if (this.keepAliveFrameId) {
				cancelAnimationFrame(this.keepAliveFrameId);
			}
			this.keepAliveFrameId = requestAnimationFrame(this.handleFrameLoop);
		};
		
		handleFrame = (timestamp) => {
			this.frames.forEach((frame) =>
				frame.call(this, { delta: this.defaultTimestep, timestamp })
			);
		};
		
		getFrames = () => {
			return this.frames;
		};
		
		add = (frame, keepAlive = false) => {
			this.cancelFrame();
			if (!this.frames.includes(frame)) {
				this.frames.push(frame);
			}
			if (keepAlive && !this.keepAliveFrames.includes(frame)) {
				this.keepAliveFrames.push(frame);
			}
			this.start();
			
			return this;
		};
		
		start = () => {
			this.frameId = requestAnimationFrame(this.handleFrame);
			this.keepAliveFrameId = requestAnimationFrame(this.handleFrameLoop);
		};
		
		cancelFrame = () => {
			if (this.frameId != null) {
				cancelAnimationFrame(this.frameId);
				this.frameId = null;
			}
			if (this.keepAliveFrameId != null) {
				cancelAnimationFrame(this.keepAliveFrameId);
				this.keepAliveFrameId = null;
			}
		};
		
		stopFrame = (frames, frame) => {
			const taskIndex = frames.indexOf(frame);
			if (taskIndex !== -1) {
				frames.splice(taskIndex, 1);
			}
			if (frames.length === 0) {
				this.cancelFrame();
				this.lastTimestamp = null;
			}
		};
		
		remove = (frame) => {
			this.stopFrame(this.frames, frame);
			this.stopFrame(this.keepAliveFrames, frame);
			
			return this;
		};
		
		clear = () => {
			this.frames = [];
			this.keepAliveFrames = [];
			this.cancelFrame();
			this.lastTimestamp = null;
			
			return this;
		};
	}
	
	const frameManager = new FrameManager();
	
	function interpolate({
		                     inputRange,
		                     outputRange,
		                     value,
		                     easing = (value) => value,
		                     reverseEasing = false,
	                     }) {
		const sortedRanges = inputRange
			.map((_, i) => ({ input: inputRange[i], output: outputRange[i] }))
			.sort((a, b) => a.input - b.input);
		const sortedInputRange = sortedRanges.map(({ input }) => input);
		const sortedOutputRange = sortedRanges.map(({ output }) => output);
		
		if (value <= sortedInputRange[0]) {
			return sortedOutputRange[0];
		}
		
		if (value >= sortedInputRange[sortedInputRange.length - 1]) {
			return sortedOutputRange[sortedOutputRange.length - 1];
		}
		
		let i = 0;
		for (const inputValue of sortedInputRange) {
			if (inputValue < value) {
				i++;
			}
		}
		
		const j = i - 1;
		
		let ratio =
			(value - sortedInputRange[j]) / (sortedInputRange[i] - sortedInputRange[j]);
		if (typeof easing === "function") {
			if (reverseEasing) {
				ratio = 1 - easing(1 - ratio);
			} else {
				ratio = easing(ratio);
			}
		}
		return sortedOutputRange[j] * (1 - ratio) + sortedOutputRange[i] * ratio;
	}
	
	const isMobile = {
		android: navigator.userAgent.match(/Android/i),
		blackBerry: navigator.userAgent.match(/BlackBerry/i),
		ipad: navigator.userAgent.match(/iPad/i),
		iOS: navigator.userAgent.match(/iPhone|iPad|iPod/i),
		opera: navigator.userAgent.match(/Opera Mini/i),
		windows: navigator.userAgent.match(/Windows Phone/i),
		amazonePhone: navigator.userAgent.match(
			/(?:SD4930UR|\\bSilk(?:.+)Mobile\\b)/i
		),
		amazoneTablet: navigator.userAgent.match(/Silk/i),
		any: navigator.userAgent.match(
			/Android|BlackBerry|iPhone|iPad|iPod|Opera Mini|Windows Phone|(?:SD4930UR|\bSilk(?:.+)Mobile\b)|Silk/i
		),
	};
	
	class ParallaxScrollItem {
		constructor(el, options, contextOptions) {
			this.el = el;
			this.defaultStyle = this.el.style.cssText;
			this.options = options;
			this.contextOptions = contextOptions;
			this.currentValue = 0;
			this.targetValue = 0;
			frameManager.add(this.handleFrameSyncUpdate, true);
		}
		
		getInputRange() {
			const { keyframes } = this.options;
			return reduce(
				objectKeys(keyframes),
				(arr, item) => {
					const val = Number(item.replace("%", ""));
					if (isNaN(val)) {
						return arr;
					}
					return [...arr, val];
				},
				[]
			).sort((a, b) => a - b);
		}
		
		getKeyframesByProp(prop) {
			const { keyframes } = this.options;
			const inputRange = this.getInputRange();
			if (Math.max(...inputRange) > 100) {
				throwError("Max value of input range must be less than 100%");
			}
			let prevVal = null;
			return reduce(
				inputRange,
				(arr, item) => {
					const key = `${item}%`;
					const val = keyframes[key][prop];
					if (val != null) {
						arr.push(String(val));
					}
					prevVal = arr[arr.length - 1];
					if (val == null && prevVal != null) {
						arr.push(prevVal);
					}
					return arr;
				},
				[]
			);
		}
		
		getDefaultUnit(prop) {
			switch (prop) {
				case "x":
				case "y":
				case "width":
				case "height":
				case "backgroundPositionY":
					return "px";
				case "rotate":
				case "rotateX":
				case "rotateY":
				case "skew":
				case "skewX":
				case "skewY":
					return "deg";
				case "scale":
				case "scaleX":
				case "scaleY":
				case "opacity":
				case "videoTime":
				default:
					return "";
			}
		}
		
		interpolate(value, prop) {
			const outputRange = this.getKeyframesByProp(prop);
			const unit = String(outputRange[0]).replace(/[0-9.,-]/g, "");
			const outputRangeNumber = outputRange.map((item) =>
				Number(String(item).replace(UNIT_PATTERN, ""))
			);
			
			if (outputRange.length === 0) {
				return EMPTY;
			}
			const result = interpolate({
				value,
				inputRange: this.getInputRange(),
				outputRange: outputRangeNumber,
			});
			return `${result}${unit || this.getDefaultUnit(prop)}`;
		}
		
		setStyles(el, value) {
			const { setStyles } = this.contextOptions;
			const x = this.interpolate(value, "x");
			const y = this.interpolate(value, "y");
			const rotate = this.interpolate(value, "rotate");
			const rotateX = this.interpolate(value, "rotateX");
			const rotateY = this.interpolate(value, "rotateY");
			const scale = this.interpolate(value, "scale");
			const scaleX = this.interpolate(value, "scaleX");
			const scaleY = this.interpolate(value, "scaleY");
			const skew = this.interpolate(value, "skew");
			const skewX = this.interpolate(value, "skewX");
			const skewY = this.interpolate(value, "skewY");
			const opacity = this.interpolate(value, "opacity");
			const width = this.interpolate(value, "width");
			const height = this.interpolate(value, "height");
			const backgroundPositionY = this.interpolate(value, "backgroundPositionY");
			const videoTime = this.interpolate(value, "videoTime");
			const groupImg = this.interpolate(value, "groupImg");
			el.style.transform = filter(
				[
					x === EMPTY ? "" : `translateX(${x})`,
					y === EMPTY ? "" : `translateY(${y})`,
					rotate === EMPTY ? "" : `rotate(${rotate})`,
					rotateX === EMPTY ? "" : `rotateX(${rotateX})`,
					rotateY === EMPTY ? "" : `rotateY(${rotateY})`,
					scale === EMPTY ? "" : `scale(${scale})`,
					scaleX === EMPTY ? "" : `scaleX(${scaleX})`,
					scaleY === EMPTY ? "" : `scaleY(${scaleY})`,
					skew === EMPTY ? "" : `skew(${skew})`,
					skewX === EMPTY ? "" : `skewX(${skewX})`,
					skewY === EMPTY ? "" : `skewY(${skewY})`,
				],
				(item) => !!item && !item.includes(EMPTY)
			).join(" ");
			if (width !== EMPTY) {
				el.style.width = `${width}`;
			}
			if (height !== EMPTY) {
				el.style.height = `${height}`;
			}
			if (opacity !== EMPTY) {
				el.style.opacity = `${opacity}`;
			}
			if (backgroundPositionY !== EMPTY) {
				el.style.backgroundSize = "cover";
				el.style.backgroundPosition = "center";
				if (isMobile.iOS) {
					el.style.backgroundPosition = `50% calc(${backgroundPositionY} * -1)`;
				} else {
					el.style.backgroundAttachment = "fixed";
					el.style.backgroundPosition = `50% ${backgroundPositionY}`;
				}
			}
			if (videoTime !== EMPTY) {
				const videoEl = el.querySelector("video");
				if (videoEl && videoEl.duration) {
					const currentTime = interpolate({
						value: Number(videoTime.replace(UNIT_PATTERN, "")),
						inputRange: [0, 100],
						outputRange: [0, videoEl.duration],
					});
					videoEl.currentTime = currentTime;
				}
			}
			if (groupImg !== EMPTY) {
				if (window.getComputedStyle(el).position === "static") {
					el.style.position = "relative";
				}
				const imageEls = Array.from(el.querySelectorAll("img"));
				const currentIndex = Math.floor(
					interpolate({
						value: Number(groupImg.replace(UNIT_PATTERN, "")),
						inputRange: [0, 100],
						outputRange: [0, imageEls.length - 1],
					})
				);
				each(imageEls, (imageEl, index) => {
					imageEl.style.visibility =
						index === currentIndex ? "visible" : "hidden";
				});
			}
			
			if (setStyles) {
				setStyles({
					element: el,
					createValue: (prop) => this.interpolate(value, prop),
					EMPTY,
				});
			}
		}
		
		handleFrameSyncUpdate = ({ delta }) => {
			const { lerpEase } = this.contextOptions;
			const diff = Math.abs(this.targetValue - this.currentValue);
			
			// Don't update if difference is too low
			if (diff < 0.001) {
				return;
			}
			let slowDownFactor = delta / DT_FPS;
			const slowDownFactorRounded = Math.round(slowDownFactor);
			if (slowDownFactorRounded >= 1) {
				slowDownFactor = slowDownFactorRounded;
			}
			const value = lerp(
				this.currentValue,
				this.targetValue,
				lerpEase * slowDownFactor
			);
			this.setStyles(this.el, value);
			this.currentValue = value;
		};
		
		handleParallax() {
			const start = window.scrollY - this.getFrom();
			const end = this.getTo() - this.getFrom();
			const value = clamp((start / end) * 100, MIN, MAX);
			if (value >= MIN && value <= MAX) {
				this.targetValue = value;
			}
		}
		
		getFrom() {
			const { from } = this.options;
			if (typeof from === "function") {
				return from();
			}
			return from;
		}
		
		getTo() {
			const { to } = this.options;
			if (typeof to === "function") {
				return to();
			}
			return to;
		}
		
		destroy = () => {
			if (this.el) {
				this.el.style.cssText = this.defaultStyle;
				frameManager.remove(this.handleFrameSyncUpdate);
			}
		};
		
		init() {
			this.handleParallax();
		}
	}
	
	function objectParse(value) {
		const val = value.trim();
		if (/^{|\[/g.test(val)) {
			try {
				const fn = new Function(`return ${val}`);
				const obj = fn();
				return JSON.parse(JSON.stringify(obj));
			} catch {
				if (/^\[/g.test(val)) {
					return [];
				}
				return {};
			}
		} else {
			return {};
		}
	}
	
	function map(array, callback) {
		const result = [];
		for (let i = 0; i < array.length; i++) {
			result.push(callback(array[i], i, array));
		}
		return result;
	}
	
	function kebabToCamel(value) {
		return value.replace(/([-]\w)/g, (g) => g[1].toUpperCase());
	}
	
	function getValue(value, type) {
		switch (type) {
			case "string":
				return value;
			case "number":
				return Number(value);
			case "boolean":
				return value === "true" || value === "";
			case "object":
				return objectParse(value);
			case "array":
				return objectParse(value);
			default:
				return value;
		}
	}
	
	function getAttrs(
		el,
		{ prefix = "v-", pick, types, camelCase = true, propTransformer } = {}
	) {
		let result = {};
		
		if (el == null) {
			return result;
		}
		
		const attrs = Array.from(el.attributes);
		if (pick == null) {
			pick = map(attrs, (attr) => attr.name);
		}
		for (const attr of attrs) {
			if (!attr.name.startsWith(prefix)) {
				continue;
			}
			let name = camelCase
				? kebabToCamel(attr.name)
				: attr.name.replace(prefix, "");
			const type = types?.[name];
			if (attr.value != null && pick?.includes(name)) {
				if (typeof propTransformer === "function") {
					name = propTransformer(name);
				}
				if (!!types && type != null) {
					result = { ...result, [name]: getValue(attr.value, type) };
				} else {
					result = { ...result, [name]: attr.value };
				}
			}
		}
		return result;
	}
	
	function reduce(array, callback, initialValue) {
		let accumulator = initialValue;
		for (let i = 0; i < array.length; i++) {
			accumulator = callback(accumulator, array[i], i, array);
		}
		return accumulator;
	}
	
	const objectKeys = (obj) => {
		return Object.keys(obj);
	};
	
	function getBreakpointsOptions(breakpoints) {
		if (breakpoints == null) {
			return;
		}
		const breakpointKeys = objectKeys(breakpoints || {});
		const result = reduce(
			breakpointKeys,
			(acc, key, index) => {
				const max = parseInt(key.toString()) || 0;
				const min = parseInt(breakpointKeys[index - 1]?.toString?.() || "0");
				if (window.innerWidth >= min && window.innerWidth <= max) {
					return {
						...acc,
						...breakpoints[key],
					};
				}
				return acc;
			},
			{}
		);
		
		if (!objectKeys(result).length) {
			return;
		}
		return result;
	}
	
	function getWindow(el) {
		return el.nodeType === 9 && el.defaultView;
	}
	
	function offset(el) {
		const doc = el?.ownerDocument;
		const docElem = doc.documentElement;
		const win = getWindow(doc);
		let box = { top: 0, left: 0 };
		
		if (!doc) {
			return {
				top: 0,
				left: 0,
			};
		}
		
		if (typeof el.getBoundingClientRect !== typeof undefined) {
			box = el.getBoundingClientRect();
		}
		
		return {
			top: box.top + win.scrollY - docElem.clientTop,
			left: box.left + win.scrollX - docElem.clientLeft,
		};
	}
	
	function resizeAxis(value) {
		let prevWidowWidth = window.innerWidth;
		let prevWindowHeight = window.innerHeight;
		
		return function (...args) {
			// @ts-ignore
			// eslint-disable-next-line @typescript-eslint/no-this-alias
			const context = this;
			if (value === "x") {
				if (prevWidowWidth !== window.innerWidth) {
					fn.apply(context, args);
				}
			} else if (value === "y") {
				if (prevWindowHeight !== window.innerHeight) {
					fn.apply(context, args);
				}
			}
			prevWidowWidth = window.innerWidth;
			prevWindowHeight = window.innerHeight;
		};
	}
	
	function throwError(message, example) {
		throw new Error(`${message}\n\n${example ? `Example:\n${example}` : ""}`);
	}
	
	class VedaParallaxScroll {
		static defaultOptions = {
			targetElement: window,
			setStyles: undefined,
			lerpEase: 0.08,
		};
		
		constructor(options) {
			this.options = {
				...VedaParallaxScroll.defaultOptions,
				...options,
			};
			this.items = [];
		}
		
		handlerScroll = () => {
			for (const instance of this.items) {
				instance.init();
			}
		};
		
		add = (el, options) => {
			this.items.push(new ParallaxScrollItem(el, options, this.options));
			return this;
		};
		
		run = () => {
			const { targetElement } = this.options;
			this.handlerScroll();
			targetElement.removeEventListener("scroll", this.handlerScroll, false);
			targetElement.addEventListener("scroll", this.handlerScroll, false);
		};
		
		destroy = () => {
			const { targetElement } = this.options;
			targetElement.removeEventListener("scroll", this.handlerScroll, false);
			for (const instance of this.items) {
				instance.destroy();
			}
		};
	}
	
	function parallaxScroll(options = {}) {
		return new VedaParallaxScroll(options);
	}
	
	class ParallaxScroll extends HTMLElement {
		keyframes = {};
		
		parallax = null;
		
		get options() {
			const options = getAttrs(this, {
				pick: ["vKeyframes", "vBreakpoints", "vLerpEase"],
				types: {
					vKeyframes: "object",
					vBreakpoints: "object",
					vLerpEase: "number",
				},
			});
			
			return options;
		}
		
		setKeyframes() {
			const { vKeyframes, vBreakpoints } = this.options;
			this.keyframes = getBreakpointsOptions(vBreakpoints) ?? vKeyframes;
		}
		
		beautyAttr = () => {
			const breakpointsAttr = this.getAttribute("v-breakpoints");
			const keyframesAttr = this.getAttribute("v-keyframes");
			if (breakpointsAttr) {
				this.setAttribute(
					"v-breakpoints",
					breakpointsAttr.replace(/\s+/g, " ").trim()
				);
			}
			if (keyframesAttr) {
				this.setAttribute(
					"v-keyframes",
					keyframesAttr.replace(/\s+/g, " ").trim()
				);
			}
		};
		
		init = () => {
			const providerEl = this.closest("v-parallax");
			if (!this.getAttribute("v-keyframes")) {
				throwError(
					`The 'v-parallax-scroll' component must have the "v-keyframes" attribute`
				);
			}
			if (providerEl && providerEl.tagName.toLowerCase() === "v-parallax") {
				const { vLerpEase = 0.08 } = this.options;
				this.setKeyframes();
				this.parallax = parallaxScroll({ lerpEase: vLerpEase });
				
				this.beautyAttr();
				
				this.parallax
					.add(this, {
						from: () => {
							return offset(providerEl).top - window.innerHeight;
						},
						to: () => {
							return offset(providerEl).top + providerEl.offsetHeight;
						},
						keyframes: this.keyframes,
					})
					.run();
			}
		};
		
		handleResize = resizeAxis("x", () => {
			this.parallax?.destroy();
			this.init();
		});
		
		connectedCallback() {
			this.init();
			window.addEventListener("resize", this.handleResize);
		}
		
		disconnectedCallback() {
			this.parallax?.destroy();
			window.addEventListener("resize", this.handleResize);
		}
	}
	
	customElements.define("v-parallax-scroll", ParallaxScroll);
})()
