(function () {
	"use strict";
	
	const DEFAULT_FPS = 60;
	const DT_FPS = 1000 / DEFAULT_FPS;
	
	const easings = {
		linear: (t) => t,
		ease: (t) => 0.5 * (1 - Math.cos(Math.PI * t)),
		easeInQuad: (t) => t * t,
		easeOutQuad: (t) => t * (2 - t),
		easeInOutQuad: (t) => (t < 0.5 ? 2 * t * t : -1 + (4 - 2 * t) * t),
		easeInCubic: (t) => t * t * t,
		easeOutCubic: (t) => --t * t * t + 1,
		easeInOutCubic: (t) =>
			t < 0.5 ? 4 * t * t * t : (t - 1) * (2 * t - 2) * (2 * t - 2) + 1,
		easeInQuart: (t) => t * t * t * t,
		easeOutQuart: (t) => 1 - --t * t * t * t,
		easeInOutQuart: (t) =>
			t < 0.5 ? 8 * t * t * t * t : 1 - 8 * --t * t * t * t,
		easeInQuint: (t) => t * t * t * t * t,
		easeOutQuint: (t) => 1 + --t * t * t * t * t,
		easeInOutQuint: (t) =>
			t < 0.5 ? 16 * t * t * t * t * t : 1 + 16 * --t * t * t * t * t,
		easeOutBounce: (t) => {
			const n1 = 7.5625;
			const d1 = 2.75;
			
			if (t < 1 / d1) {
				return n1 * t * t;
			}
			if (t < 2 / d1) {
				return n1 * (t -= 1.5 / d1) * t + 0.75;
			}
			if (t < 2.5 / d1) {
				return n1 * (t -= 2.25 / d1) * t + 0.9375;
			}
			return n1 * (t -= 2.625 / d1) * t + 0.984375;
		},
		easeInBounce: (t) => 1 - easings.easeOutBounce(1 - t),
		easeOutBack: (t) => {
			const c1 = 1.70158;
			const c3 = c1 + 1;
			
			return 1 + c3 * (t - 1) ** 3 + c1 * (t - 1) ** 2;
		},
		easeInBack: (t) => {
			const c1 = 1.70158;
			const c3 = c1 + 1;
			
			return c3 * t * t * t - c1 * t * t;
		},
		easeInOut: (t) =>
			t < 0.5
				? easings.easeInBack(t * 2) / 2
				: easings.easeOutBack(t * 2 - 1) / 2 + 0.5,
		easeInElastic: (t) => {
			const c4 = (2 * Math.PI) / 3;
			return t === 0
				? 0
				: t === 1
					? 1
					: -(2 ** (10 * t - 10)) * Math.sin((t * 10 - 10.75) * c4);
		},
		easeOutElastic: (t) => {
			const c4 = (2 * Math.PI) / 3;
			return t === 0
				? 0
				: t === 1
					? 1
					: 2 ** (-10 * t) * Math.sin((t * 10 - 0.75) * c4) + 1;
		},
		spring: (t) => 1 - Math.cos(t * 4.5 * Math.PI) * Math.exp(-t * 6),
		decay: (t) => 1 - Math.exp(-t * 6),
		bezier: (x1, y1, x2, y2) => (t) => {
			const ax = 3 * x1 - 3 * x2 + 1;
			const bx = 3 * x2 - 6 * x1;
			const cx = 3 * x1;
			const ay = 3 * y1 - 3 * y2 + 1;
			const by = 3 * y2 - 6 * y1;
			const cy = 3 * y1;
			
			// eslint-disable-next-line @typescript-eslint/no-shadow
			const sampleCurveX = (t) => ((ax * t + bx) * t + cx) * t;
			
			const sampleCurveY = (t) => ((ay * t + by) * t + cy) * t;
			
			const sampleCurveDerivativeX = (t) => (3 * ax * t + 2 * bx) * t + cx;
			
			const solveCurveX = (x, epsilon) => {
				let t0;
				let t1;
				let t2;
				let x2;
				let d2;
				let i;
				
				for (t2 = x, i = 0; i < 8; i++) {
					x2 = sampleCurveX(t2) - x;
					if (Math.abs(x2) < epsilon) {
						return t2;
					}
					d2 = sampleCurveDerivativeX(t2);
					if (Math.abs(d2) < 1e-6) {
						break;
					}
					t2 -= x2 / d2;
				}
				
				t0 = 0;
				t1 = 1;
				t2 = x;
				
				if (t2 < t0) {
					return t0;
				}
				if (t2 > t1) {
					return t1;
				}
				
				while (t0 < t1) {
					x2 = sampleCurveX(t2);
					if (Math.abs(x2 - x) < epsilon) {
						return t2;
					}
					if (x > x2) {
						t0 = t2;
					} else {
						t1 = t2;
					}
					t2 = (t1 - t0) * 0.5 + t0;
				}
				
				return t2;
			};
			
			return sampleCurveY(solveCurveX(t, 1 / 200));
		},
	};
	
	function interpolate({
		                     inputRange,
		                     outputRange,
		                     value,
		                     easing = (value) => value,
		                     reverseEasing = false,
	                     }) {
		const sortedRanges = inputRange
			.map((_, i) => ({input: inputRange[i], output: outputRange[i]}))
			.sort((a, b) => a.input - b.input);
		const sortedInputRange = sortedRanges.map(({input}) => input);
		const sortedOutputRange = sortedRanges.map(({output}) => output);
		
		if (value <= sortedInputRange[0]) {
			return sortedOutputRange[0];
		}
		
		if (value >= sortedInputRange[sortedInputRange.length - 1]) {
			return sortedOutputRange[sortedOutputRange.length - 1];
		}
		
		let i = 0;
		for (const inputValue of sortedInputRange) {
			if (inputValue < value) {
				i++;
			}
		}
		
		const j = i - 1;
		
		let ratio =
			(value - sortedInputRange[j]) / (sortedInputRange[i] - sortedInputRange[j]);
		if (typeof easing === "function") {
			if (reverseEasing) {
				ratio = 1 - easing(1 - ratio);
			} else {
				ratio = easing(ratio);
			}
		}
		return sortedOutputRange[j] * (1 - ratio) + sortedOutputRange[i] * ratio;
	}
	
	function getWindow(el) {
		return el.nodeType === 9 && el.defaultView;
	}
	
	function offset(el) {
		const doc = el?.ownerDocument;
		const docElem = doc.documentElement;
		const win = getWindow(doc);
		let box = {top: 0, left: 0};
		
		if (!doc) {
			return {
				top: 0,
				left: 0,
			};
		}
		
		if (typeof el.getBoundingClientRect !== typeof undefined) {
			box = el.getBoundingClientRect();
		}
		
		return {
			top: box.top + win.scrollY - docElem.clientTop,
			left: box.left + win.scrollX - docElem.clientLeft,
		};
	}
	
	function objectParse(value) {
		const val = value.trim();
		if (/^{|\[/g.test(val)) {
			try {
				const fn = new Function(`return ${val}`);
				const obj = fn();
				return JSON.parse(JSON.stringify(obj));
			} catch {
				if (/^\[/g.test(val)) {
					return [];
				}
				return {};
			}
		} else {
			return {};
		}
	}
	
	const lerp = (p1, p2, t) => {
		return p1 + (p2 - p1) * t;
	};
	
	function map(array, callback) {
		const result = [];
		for (let i = 0; i < array.length; i++) {
			result.push(callback(array[i], i, array));
		}
		return result;
	}
	
	function kebabToCamel(value) {
		return value.replace(/([-]\w)/g, (g) => g[1].toUpperCase());
	}
	
	function getValue(value, type) {
		switch (type) {
			case "string":
				return value;
			case "number":
				return Number(value);
			case "boolean":
				return value === "true" || value === "";
			case "object":
				return objectParse(value);
			case "array":
				return objectParse(value);
			default:
				return value;
		}
	}
	
	function getAttrs(
		el,
		{prefix = "v-", pick, types, camelCase = true, propTransformer} = {}
	) {
		let result = {};
		
		if (el == null) {
			return result;
		}
		
		const attrs = Array.from(el.attributes);
		if (pick == null) {
			pick = map(attrs, (attr) => attr.name);
		}
		for (const attr of attrs) {
			if (!attr.name.startsWith(prefix)) {
				continue;
			}
			let name = camelCase
				? kebabToCamel(attr.name)
				: attr.name.replace(prefix, "");
			const type = types?.[name];
			if (attr.value != null && pick?.includes(name)) {
				if (typeof propTransformer === "function") {
					name = propTransformer(name);
				}
				if (!!types && type != null) {
					result = {...result, [name]: getValue(attr.value, type)};
				} else {
					result = {...result, [name]: attr.value};
				}
			}
		}
		return result;
	}
	
	class FrameManager {
		frameId;
		keepAliveFrameId;
		lastTimestamp;
		frames;
		keepAliveFrames;
		defaultTimestep;
		
		constructor() {
			this.frameId = null;
			this.keepAliveFrameId = null;
			this.lastTimestamp = null;
			this.frames = [];
			this.keepAliveFrames = [];
			this.defaultTimestep = (1 / 60) * 1000;
		}
		
		handleFrameLoop = (timestamp) => {
			if (this.lastTimestamp) {
				const delta = timestamp - this.lastTimestamp;
				this.keepAliveFrames.forEach((frame) =>
					frame.call(this, {delta, timestamp})
				);
			}
			this.lastTimestamp = timestamp;
			if (this.keepAliveFrameId) {
				cancelAnimationFrame(this.keepAliveFrameId);
			}
			this.keepAliveFrameId = requestAnimationFrame(this.handleFrameLoop);
		};
		
		handleFrame = (timestamp) => {
			this.frames.forEach((frame) =>
				frame.call(this, {delta: this.defaultTimestep, timestamp})
			);
		};
		
		getFrames = () => {
			return this.frames;
		};
		
		add = (frame, keepAlive = false) => {
			this.cancelFrame();
			if (!this.frames.includes(frame)) {
				this.frames.push(frame);
			}
			if (keepAlive && !this.keepAliveFrames.includes(frame)) {
				this.keepAliveFrames.push(frame);
			}
			this.start();
			
			return this;
		};
		
		start = () => {
			this.frameId = requestAnimationFrame(this.handleFrame);
			this.keepAliveFrameId = requestAnimationFrame(this.handleFrameLoop);
		};
		
		cancelFrame = () => {
			if (this.frameId != null) {
				cancelAnimationFrame(this.frameId);
				this.frameId = null;
			}
			if (this.keepAliveFrameId != null) {
				cancelAnimationFrame(this.keepAliveFrameId);
				this.keepAliveFrameId = null;
			}
		};
		
		stopFrame = (frames, frame) => {
			const taskIndex = frames.indexOf(frame);
			if (taskIndex !== -1) {
				frames.splice(taskIndex, 1);
			}
			if (frames.length === 0) {
				this.cancelFrame();
				this.lastTimestamp = null;
			}
		};
		
		remove = (frame) => {
			this.stopFrame(this.frames, frame);
			this.stopFrame(this.keepAliveFrames, frame);
			
			return this;
		};
		
		clear = () => {
			this.frames = [];
			this.keepAliveFrames = [];
			this.cancelFrame();
			this.lastTimestamp = null;
			
			return this;
		};
	}
	
	const frameManager = new FrameManager();
	
	class ParallaxHover extends HTMLElement {
		shadow = this.attachShadow({mode: "open"});
		
		providerEl = null;
		
		targetValueX = 0;
		
		targetValueY = 0;
		
		currentValueX = 0;
		
		currentValueY = 0;
		
		static defaultOptions = {
			vStrength: 100,
			vResetPosition: true,
			v3d: false,
		};
		
		constructor() {
			super();
			frameManager.add(this.handleFrameSyncUpdate, true);
		}
		
		get options() {
			const options = getAttrs(this, {
				pick: ["vStrength", "vResetPosition", "v3d"],
				types: {
					vStrength: "number",
					vResetPosition: "boolean",
					v3d: "boolean",
				},
			});
			
			return {
				...ParallaxHover.defaultOptions,
				...options,
			};
		}
		
		handleParallax = (delta) => {
			const {vStrength, v3d} = this.options;
			let slowDown = delta / DT_FPS;
			const slowDownRounded = Math.round(slowDown);
			if (slowDownRounded >= 1) {
				slowDown = slowDownRounded;
			}
			const lerpEase = Number(
				this.providerEl?.getAttribute("v-lerp-ease") || "0.08"
			);
			const valueX = lerp(
				this.currentValueX,
				this.targetValueX,
				lerpEase * slowDown
			);
			const valueY = lerp(
				this.currentValueY,
				this.targetValueY,
				lerpEase * slowDown
			);
			const innerEl = this.shadow.querySelector("v-parallax-hover-inner");
			if (v3d) {
				const constant = 4;
				innerEl.style.transformStyle = "preserve-3d";
				innerEl.style.transform = `perspective(2000px) rotateX(${
					(-valueY / constant) * vStrength
				}deg) rotateY(${(valueX / constant) * vStrength}deg)`;
			} else {
				innerEl.style.transform = `translate3d(${valueX * vStrength}px, ${
					valueY * vStrength
				}px, 0)`;
			}
			this.currentValueX = valueX;
			this.currentValueY = valueY;
		};
		
		handleFrameSyncUpdate = ({delta}) => {
			const diffX = Math.abs(this.targetValueX - this.currentValueX);
			const diffY = Math.abs(this.targetValueY - this.currentValueY);
			if (diffX < 0.001 && diffY < 0.001) {
				return;
			}
			
			this.handleParallax(delta);
		};
		
		setTargetValue = (event, left, top, width, height) => {
			const centerX = offset(this).left + this.offsetWidth / 2;
			const centerY = offset(this).top + this.offsetHeight / 2;
			this.targetValueX = interpolate({
				value: event.pageX - left,
				inputRange: [0, centerX - left, width],
				outputRange: [-1, 0, 1],
			});
			this.targetValueY = interpolate({
				value: event.pageY - top,
				inputRange: [0, centerY - top, height],
				outputRange: [-1, 0, 1],
			});
		};
		
		handleMouseMove = (event) => {
			const providerLeft = offset(this.providerEl).left;
			const providerTop = offset(this.providerEl).top;
			const providerWidth = this.providerEl.offsetWidth;
			const providerHeight = this.providerEl.offsetHeight;
			this.setTargetValue(
				event,
				providerLeft,
				providerTop,
				providerWidth,
				providerHeight
			);
		};
		
		handleMouseLeave = () => {
			const {vResetPosition} = this.options;
			if (vResetPosition) {
				this.targetValueX = 0;
				this.targetValueY = 0;
			}
		};
		
		connectedCallback() {
			this.providerEl = this.closest("v-parallax");
			if (!this.providerEl) {
				throw new Error(
					`The v-parallax-hover component must be a child of v-parallax`
				);
			}
			if (!this.shadow.innerHTML) {
				this.shadow.innerHTML = `<v-parallax-hover-inner style="display: block"><slot></slot></v-parallax-hover-inner>`;
			}
			this.handleParallax((1 / 60) * 1000);
			this.providerEl.addEventListener("mousemove", this.handleMouseMove);
			this.providerEl.addEventListener("mouseleave", this.handleMouseLeave);
		}
		
		disconnectedCallback() {
			frameManager.remove(this.handleFrameSyncUpdate);
			this.providerEl?.removeEventListener("mousemove", this.handleMouseMove);
			this.providerEl?.removeEventListener("mouseleave", this.handleMouseLeave);
		}
	}
	
	customElements.define("v-parallax-hover", ParallaxHover);
	
})();
